// -------------------------------------------------------------
// variables that are provided by the application
// -------------------------------------------------------------
float4x4 g_view_projection_matrix;	
float4x4 g_world_matrix;	
float4 vecLightDir = float4(0.0f, -1.0f, 0.0f, 1.0f);
float4 g_camera_eye_vector;

texture g_tex0;
sampler ColorMapSampler = sampler_state
{
   Texture = <g_tex0>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;   
   AddressU  = Clamp;
   AddressV  = Clamp;
};

texture g_tex1;
sampler BumpMapSampler = sampler_state
{
   Texture = <g_tex1>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;   
   AddressU  = Clamp;
   AddressV  = Clamp;
};

// -------------------------------------------------------------
// Output channels
// -------------------------------------------------------------
struct VS_OUTPUT
{
    float4 Pos  : POSITION;
    float2 Tex : TEXCOORD0;
    float3 Light : TEXCOORD1;
    float3 View : TEXCOORD2;
    float3 worldPos : TEXCOORD3;
};

// -------------------------------------------------------------
// vertex shader function (input channels)
// -------------------------------------------------------------
VS_OUTPUT VS(float4 Pos : POSITION, float2 Tex : TEXCOORD, float3 Normal : NORMAL, float3 Tangent : TANGENT  )
{
    VS_OUTPUT Out = (VS_OUTPUT)0;      
    Out.Pos = mul(Pos, g_world_matrix);	// transform Position
    Out.Pos = mul(Out.Pos, g_view_projection_matrix);
    
    // compute the 3x3 tranform matrix 
    // to transform from world space to tangent space
    float3x3 worldToTangentSpace;
    worldToTangentSpace[0] = mul(Tangent, g_world_matrix);
    worldToTangentSpace[1] = mul(cross(Tangent, Normal), g_world_matrix);
    worldToTangentSpace[2] = mul(Normal, g_world_matrix);
        
    Out.Tex = Tex;
    
    float4 PosWorld = mul(Pos, g_world_matrix);	
    Out.worldPos = PosWorld;
    Out.Light = mul(worldToTangentSpace, vecLightDir); 	// L
    Out.View = mul(worldToTangentSpace, g_camera_eye_vector - PosWorld);	// V
    
   return Out;
}

// -------------------------------------------------------------
// Pixel Shader (input channels):output channel
// -------------------------------------------------------------
float4 PS(float2 Tex: TEXCOORD0, float3 Light : TEXCOORD1, float3 View : TEXCOORD2, float3 worldPos : TEXCOORD3 ) : COLOR
{
    // fetch color map
    float4 Color = tex2D(ColorMapSampler, Tex);	
    
    // bump map, recover tangent space normal by -1
    //from [0..1] to [-1..1]
    float3 Normal =(2 * (tex2D(BumpMapSampler, Tex)))- 1.0; 
 	
    float3 LightDir = normalize(Light);	// L
    float3 ViewDir = normalize(View);	// V
        
	// diffuse component
    float Diffuse = saturate(dot(Normal, LightDir));
    
    // reflection vector
    float3 Reflect = normalize(2 * Diffuse * Normal - LightDir);  // R
 	
    // gloss map in Color.w restricts spec reflection
    float Specular = min(pow(saturate(dot(Reflect, ViewDir)), 3), Color.w);
	

	//calculate color of the mesh, depending on the height from the floor
	float yPos = worldPos.y + 1.632f;
	//float amountRed 	= 1.0f * ( yPos * 0.06f );	
	//float amountGreen 	= 1.0f * abs((( yPos * 0.01f ) - 1.0f));
	float amountRed 	= 1.0f * ( yPos * 0.048f );	
	float amountGreen 	= 1.0f * abs((( yPos * 0.006f ) - 1.0f));
	amountGreen = amountGreen - amountRed;
	
	if( amountGreen < 0.0f )
		amountGreen = 0.0f;
	else if( amountGreen > 1.0f )
		amountGreen = 1.0f;
		
	if( amountRed < 0.0f )
		amountRed = 0.0f;
	else if( amountRed > 1.0f )
		amountRed = 1.0f;
	//
	float4 cubeColor = float4( amountRed, amountGreen, 0.0f, 1.0f );
    
    
    return (0.2 * Color + Color * Diffuse + Specular) * cubeColor;	
}

// -------------------------------------------------------------
// 
// -------------------------------------------------------------
technique normal
{
    pass P0
    {
        Sampler[0] = (ColorMapSampler);		
        Sampler[1] = (BumpMapSampler);		
    
        // compile shaders
        VertexShader = compile vs_1_1 VS();
        PixelShader  = compile ps_2_0 PS();
    }
}
